"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FullLeaderboard = void 0;
const IntertialAnimation_1 = require("./IntertialAnimation");
const DestructionHelper_1 = require("./DestructionHelper");
const ComponentUtils_1 = require("./ComponentUtils");
const LeaderboardConstants_1 = require("./Interfaces/LeaderboardConstants");
const LeaderboardRelated_1 = require("./Interfaces/LeaderboardRelated");
const LeaderboardUtils_1 = require("./LeaderboardUtils");
const FullLeaderboardAnimations_1 = require("./FullLeaderboardAnimations");
const DateUtils_1 = require("./Modules/DateUtils");
class FullLeaderboard {
    constructor(scriptComponent, leaderboardBackgroundPrefab, leaderboardEntryPrefab, leaderboardEntryFallbackPrefab, onLeaderboardClosedCallback, onLeaderboardOpenedCallback, renderOrder, bitmojiStickerLoader, backgroundCustomization) {
        this.scriptComponent = scriptComponent;
        this.leaderboardBackgroundPrefab = leaderboardBackgroundPrefab;
        this.leaderboardEntryPrefab = leaderboardEntryPrefab;
        this.leaderboardEntryFallbackPrefab = leaderboardEntryFallbackPrefab;
        this.onLeaderboardClosedCallback = onLeaderboardClosedCallback;
        this.onLeaderboardOpenedCallback = onLeaderboardOpenedCallback;
        this.renderOrder = renderOrder;
        this.bitmojiStickerLoader = bitmojiStickerLoader;
        this.destructionHelper = new DestructionHelper_1.DestructionHelper();
        this.activeTouchId = null;
        this.entriesParent = null;
        this.userRecords = [];
        this.currentUserRecord = null;
        this.entryWrappers = [];
        this.PRELOAD_ENTRY_COUNT = LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.PRELOAD_ENTRY_COUNT;
        this.appliedScroll = vec2.zero();
        this.isHeaderEnabled = false;
        this.isFallbackInitialised = false;
        this.fallbackEntriesParent = null;
        this.isLeaderboardOpened = false;
        this.outOfLeaderboardInteraction = null;
        this.animations = null;
        this.isGlobalLeaderboard = false;
        this.screenTransform = (0, ComponentUtils_1.getOrCreateScreenTransform)(this.scriptComponent.getSceneObject(), this.destructionHelper);
        this.screenTransform.anchors.setCenter(LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.INITIAL_PARENT_POSITION);
        this.scrollAnimation = this.createInertialAnimation(scriptComponent, LeaderboardConstants_1.LeaderboardConstants.InertialAnimation.MAX_SPEED, LeaderboardConstants_1.LeaderboardConstants.InertialAnimation.FRICTION, LeaderboardConstants_1.LeaderboardConstants.InertialAnimation.SNAP_DURATION, LeaderboardConstants_1.LeaderboardConstants.InertialAnimation.SPRINGNESS);
        this.scrollAnimation.enabled = false;
        const leaderboardBackground = this.leaderboardBackgroundPrefab.instantiate(scriptComponent.getSceneObject());
        leaderboardBackground.enabled = true;
        this.leaderboardBackground = leaderboardBackground.getComponent("ScriptComponent");
        this.leaderboardBackground.setHeaderMaskTopAnchors(LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.NO_HEADER_MASK_TOP_ANCHORS);
        this.leaderboardBackground.setHeaderAnchors(LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.PARENT_NO_HEADER_ANCHORS);
        this.leaderboardBackground.setBackgroundCustomizationOptionEnabled(backgroundCustomization);
        this.animations = new FullLeaderboardAnimations_1.FullLeaderboardAnimations(this.screenTransform, this.leaderboardBackground.getTexturePreviewScreenTransform(), this.leaderboardBackground.getTopBorderScreenTransform(), this.leaderboardBackground.getRoundedCornersTransform(), this.leaderboardBackground.getTexturePreviewParentScreenTransform());
        const parentScreenTransform = this.leaderboardBackground.getEntriesParentScreenTransform();
        this.entriesParent = parentScreenTransform.getSceneObject();
        this.initClosingButton();
        this.initInteraction();
        this.leaderboardBackground.setPostEffectsEnabled(false);
    }
    setLeaderboardName(name) {
        this.leaderboardBackground.setLeaderboardName(name);
    }
    setBitmoji(bitmojiTexture) {
        this.leaderboardBackground.setFixedSizedTextureHeader(bitmojiTexture);
    }
    setCustomTextureHeader(customTexture) {
        const isScreenTexture = customTexture.control.isOfType("Provider.ScreenTextureProvider");
        this.animations.setIsScreenTextureInPreview(isScreenTexture);
        if (isScreenTexture) {
            this.leaderboardBackground.setResisableTextureHeader(customTexture);
        }
        else {
            this.leaderboardBackground.setFixedSizedTextureHeader(customTexture);
            this.leaderboardBackground.setBackgroundCustomizationOptionEnabled(LeaderboardRelated_1.BackgroundCustomization.Bitmoji);
        }
    }
    show() {
        this.animateLeaderboard(true);
    }
    hide() {
        this.animateLeaderboard(false);
    }
    visualiseEntries(leaderboardRecordsWithCurrentUser, currentUserRecord) {
        this.userRecords = leaderboardRecordsWithCurrentUser;
        this.currentUserRecord = currentUserRecord;
        this.setFallbackEnabled(leaderboardRecordsWithCurrentUser.length <= 0);
        const entryWrappersNeededCount = Math.min(this.PRELOAD_ENTRY_COUNT, leaderboardRecordsWithCurrentUser.length);
        this.fillEntriesWrappersToSize(entryWrappersNeededCount);
        if (!isNull(currentUserRecord)) {
            this.enableHeaderIfPossible(currentUserRecord);
        }
        this.entryWrappers.forEach((entryWrapper, i) => {
            entryWrapper.representingIndex = i;
            this.updateEntryData(entryWrapper, this.userRecords[i]);
        });
    }
    setFallbackEnabled(enabled) {
        if (enabled) {
            if (!this.isFallbackInitialised) {
                this.initialiseFallback();
            }
            this.leaderboardBackground.setHeaderEnabled(false);
            this.leaderboardBackground.setHeaderMaskTopAnchors(LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.NO_HEADER_MASK_TOP_ANCHORS);
            this.leaderboardBackground.setHeaderAnchors(LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.PARENT_NO_HEADER_ANCHORS);
        }
        if (!isNull(this.fallbackEntriesParent)) {
            this.fallbackEntriesParent.enabled = enabled;
        }
    }
    setTimerUiEnabled(enabled) {
        this.leaderboardBackground.setTimerUiEnabled(enabled);
    }
    setTimerText(text) {
        this.leaderboardBackground.setResetText(text);
    }
    setTimeLeft(miliseconds) {
        this.leaderboardBackground.setTimerTexture(miliseconds < 1000 * 60 * 60 ? LeaderboardRelated_1.TimerTextureType.Red : LeaderboardRelated_1.TimerTextureType.Regular);
        this.leaderboardBackground.setTimeLeftText((0, DateUtils_1.convertMilisecondsToDateString)(miliseconds));
    }
    setIsGlobalLeaderboard(isGlobalLeaderboard) {
        this.isGlobalLeaderboard = isGlobalLeaderboard;
    }
    getRealIndex(representingIndex, userRecord) {
        if (!this.isGlobalLeaderboard) {
            return representingIndex + 1;
        }
        if (!isNull(userRecord) && !isNull(userRecord.globalExactRank) && userRecord.globalExactRank !== 0) {
            return userRecord.globalExactRank;
        }
        return representingIndex + 1;
    }
    initClosingButton() {
        const closingButtonInteraction = this.getOrCreateInteraction(this.leaderboardBackground.getClosingZoneScreenTransform().getSceneObject());
        closingButtonInteraction.isFilteredByDepth = true;
        let currentSameDirectionTouchCount = 0;
        let previousPos = null;
        closingButtonInteraction.onTouchStart.add((args) => {
            currentSameDirectionTouchCount = 0;
            previousPos = null;
        });
        closingButtonInteraction.onTouchMove.add((args) => {
            if (previousPos === null) {
                previousPos = args.position;
                return;
            }
            const isClosingDirection = previousPos.y < args.position.y;
            currentSameDirectionTouchCount = isClosingDirection ? currentSameDirectionTouchCount + 1 : 0;
            if (currentSameDirectionTouchCount >= LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.TOUCH_COUNT_TO_CLOSE_VIEW) {
                if (!this.animations.isPlaying()) {
                    this.animateLeaderboard(false);
                }
            }
            previousPos = args.position;
        });
    }
    fillEntriesWrappersToSize(arraySize) {
        for (let i = this.entryWrappers.length; i < arraySize; i++) {
            this.entryWrappers.push(this.createEntry(i));
        }
        this.checkIfLeaderboardScrollable();
    }
    createEntry(entryIndex) {
        const currentEntriesCount = this.entriesParent.children.length;
        const topRelativeLimit = LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRIES_LIMIT_TOP_POSITION_Y;
        const bottomPosition = topRelativeLimit - (currentEntriesCount + 1) * LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y;
        const topPosition = topRelativeLimit - currentEntriesCount * LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y;
        const entry = this.leaderboardEntryPrefab.instantiate(this.entriesParent);
        entry.enabled = true;
        (0, ComponentUtils_1.setRenderOrderRecursivelyRelativeToParent)(entry, this.renderOrder);
        (0, ComponentUtils_1.setRenderLayerRecursively)(entry, this.entriesParent.layer);
        const screenTransform = entry.getComponent("ScreenTransform");
        screenTransform.anchors.top = topPosition;
        screenTransform.anchors.bottom = bottomPosition;
        const entryWrapper = {
            sceneObject: entry,
            leaderboardEntry: entry.getComponent("ScriptComponent"),
            screenTransform: screenTransform,
            representingIndex: entryIndex,
            isCurrentUser: false,
            bitmojiRequestId: null
        };
        return entryWrapper;
    }
    getOnComplete(toShow) {
        if (!toShow) {
            return () => {
                var _a;
                (_a = this.onLeaderboardClosedCallback) === null || _a === void 0 ? void 0 : _a.call(this);
                this.outOfLeaderboardInteraction.enabled = false;
                this.isLeaderboardOpened = false;
            };
        }
        else {
            return () => {
                var _a;
                this.isLeaderboardOpened = true;
                this.outOfLeaderboardInteraction.enabled = true;
                (_a = this.onLeaderboardOpenedCallback) === null || _a === void 0 ? void 0 : _a.call(this);
            };
        }
    }
    animateLeaderboard(toShow) {
        this.animations.playAnimation(toShow, this.getOnComplete(toShow));
        this.leaderboardBackground.setPostEffectsEnabled(toShow);
    }
    checkIfEntriesShouldBeShuffled(isScrollToTop) {
        if (this.entryWrappers.length < 1) {
            return;
        }
        let lastEntry = this.entryWrappers[this.entryWrappers.length - 1];
        const indexAboveLimit = Math.floor(this.appliedScroll.y / LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y);
        if (isScrollToTop) {
            while (indexAboveLimit > this.entryWrappers[0].representingIndex && lastEntry.representingIndex < this.userRecords.length - 1) {
                this.moveEntryFromHeadToTail(lastEntry);
                lastEntry = this.entryWrappers[this.entryWrappers.length - 1];
            }
        }
        else {
            let bottomEntryYCoord = lastEntry.screenTransform.anchors.getCenter().y;
            while (bottomEntryYCoord < LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.MIN_Y_COORD_TO_SHOW_ENTRY && this.entryWrappers[0].representingIndex > 0) {
                this.moveEntryFromTailToHead(this.entryWrappers[0]);
                bottomEntryYCoord = lastEntry.screenTransform.anchors.getCenter().y;
            }
        }
    }
    moveEntryFromTailToHead(headEntry) {
        const outOfBoundEntry = this.entryWrappers.pop();
        const positionToSet = this.entryWrappers.length > 0 ?
            headEntry.screenTransform.anchors.getCenter().add(new vec2(0, LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y))
            : outOfBoundEntry.screenTransform.anchors.getCenter();
        this.entryWrappers.unshift(outOfBoundEntry);
        outOfBoundEntry.representingIndex = headEntry.representingIndex - 1;
        outOfBoundEntry.screenTransform.anchors.setCenter(positionToSet);
        this.updateEntryData(outOfBoundEntry, this.userRecords[outOfBoundEntry.representingIndex]);
    }
    moveEntryFromHeadToTail(tailEntry) {
        const outOfBoundEntry = this.entryWrappers.shift();
        const positionToSet = this.entryWrappers.length > 0 ?
            tailEntry.screenTransform.anchors.getCenter().sub(new vec2(0, LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y))
            : outOfBoundEntry.screenTransform.anchors.getCenter();
        this.entryWrappers.push(outOfBoundEntry);
        outOfBoundEntry.representingIndex = tailEntry.representingIndex + 1;
        outOfBoundEntry.screenTransform.anchors.setCenter(positionToSet);
        this.updateEntryData(outOfBoundEntry, this.userRecords[outOfBoundEntry.representingIndex]);
    }
    updateEntryData(entry, userRecord) {
        entry.leaderboardEntry.setEntryIndex(this.getRealIndex(entry.representingIndex, userRecord));
        entry.leaderboardEntry.setUserRecord(userRecord, this.isGlobalLeaderboard);
        this.bitmojiStickerLoader.dismissToken(entry.bitmojiRequestId);
        if (!isNull(userRecord.snapchatUser)) {
            entry.bitmojiRequestId = this.bitmojiStickerLoader.generateToken();
            this.bitmojiStickerLoader.loadForUser(userRecord.snapchatUser, '').then((texture) => {
                if (this.bitmojiStickerLoader.isTokenValid(entry.bitmojiRequestId)) {
                    entry.leaderboardEntry.setBitmoji(texture);
                }
            })
                .catch(() => {
                entry.leaderboardEntry.setBitmoji(null);
            });
        }
        else {
            entry.leaderboardEntry.setBitmoji(null);
        }
        const isCurrentUserEntry = (0, LeaderboardUtils_1.isSameUserEntry)(userRecord, this.currentUserRecord);
        if (entry.representingIndex === this.userRecords.length - 1) {
            if (isCurrentUserEntry) {
                entry.leaderboardEntry.setEntryBgTextureType(LeaderboardRelated_1.LeaderboardEntryBgTextureType.BottomSegmentBlue);
            }
            else {
                entry.leaderboardEntry.setEntryBgTextureType(LeaderboardRelated_1.LeaderboardEntryBgTextureType.BottomSegment);
            }
        }
        else {
            if (isCurrentUserEntry) {
                entry.leaderboardEntry.setEntryBgTextureType(LeaderboardRelated_1.LeaderboardEntryBgTextureType.MainSegmentBlue);
            }
            else {
                entry.leaderboardEntry.setEntryBgTextureType(LeaderboardRelated_1.LeaderboardEntryBgTextureType.MainSegment);
            }
        }
    }
    checkIfLeaderboardScrollable() {
        this.scrollAnimation.enabled = this.entryWrappers.length >= LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRIES_NEEDED_FOR_SCROLL;
    }
    enableHeaderIfPossible(currentUserRecord) {
        if ((!isNull(currentUserRecord.globalRankPercentile) && currentUserRecord.globalRankPercentile !== 0)
            || (!isNull(currentUserRecord.globalExactRank) && currentUserRecord.globalExactRank !== 0)) {
            this.leaderboardBackground.setCurrentUserRecord(currentUserRecord);
            this.leaderboardBackground.setHeaderEnabled(true);
            this.leaderboardBackground.setHeaderAnchors(LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.PARENT_HEADER_ANCHORS);
            this.leaderboardBackground.setHeaderMaskTopAnchors(LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.HEADER_MASK_TOP_ANCHORS);
        }
        else {
            this.leaderboardBackground.setHeaderMaskTopAnchors(LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.NO_HEADER_MASK_TOP_ANCHORS);
            this.leaderboardBackground.setHeaderAnchors(LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.PARENT_NO_HEADER_ANCHORS);
        }
    }
    createInertialAnimation(scriptComponent, maxSpeed, friction, snapDuration, springiness) {
        return new IntertialAnimation_1.InertialAnimation(scriptComponent, (delta) => {
            this.applyScroll(delta);
        }, () => {
        }, new vec2(0, 1), maxSpeed, friction, snapDuration);
    }
    applyScroll(delta) {
        if (this.entryWrappers.length <= 0) {
            return;
        }
        let offsetToApply = delta;
        if (delta.y > 0) {
            const bottomPositionInit = this.entryWrappers[this.entryWrappers.length - 1].screenTransform.anchors.getCenter();
            const bottomPositionAfterScroll = bottomPositionInit.add(delta);
            const bottomLimit = this.isHeaderEnabled ? LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRIES_BOTTOM_LIMIT_POSITION_HEADER : LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRIES_BOTTOM_LIMIT_POSITION_NO_HEADER;
            if (bottomPositionAfterScroll.y > bottomLimit) {
                offsetToApply = offsetToApply = new vec2(0, bottomLimit - bottomPositionInit.y);
            }
        }
        else {
            const topPositionInit = this.entryWrappers[0].screenTransform.anchors.getCenter();
            const topPositionAfterScroll = topPositionInit.add(delta);
            if (topPositionAfterScroll.y < LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRIES_TOP_LIMIT_POSITION) {
                offsetToApply = new vec2(0, LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRIES_TOP_LIMIT_POSITION - topPositionInit.y);
            }
        }
        this.applyOffset(offsetToApply);
        this.checkIfEntriesShouldBeShuffled(delta.y > 0);
    }
    applyOffset(offset) {
        this.entryWrappers.forEach((entryWrappers) => {
            let center = entryWrappers.screenTransform.anchors.getCenter();
            center = center.add(offset);
            entryWrappers.screenTransform.anchors.setCenter(center);
            if (center.y > LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.MAX_Y_COORD_TO_SHOW_ENTRY
                || center.y < LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.MIN_Y_COORD_TO_SHOW_ENTRY) {
                entryWrappers.sceneObject.enabled = false;
            }
            else {
                entryWrappers.sceneObject.enabled = true;
            }
        });
        this.appliedScroll = this.appliedScroll.add(offset);
    }
    initInteraction() {
        const interaction = this.getOrCreateInteraction(this.leaderboardBackground.getTouchZoneSceneObject());
        interaction.isFilteredByDepth = true;
        interaction.onTouchStart.add(args => {
            if (!this.isInTouchZone(args.position))
                return;
            if (this.activeTouchId == null) {
                this.activeTouchId = args.touchId;
                this.scrollAnimation.onTouchStart(this.screenPointtoLocalPoint(args.position));
            }
        });
        interaction.onTouchMove.add(args => {
            if (this.activeTouchId == args.touchId) {
                this.scrollAnimation.onTouchMove(this.screenPointtoLocalPoint(args.position));
            }
        });
        interaction.onTouchEnd.add(args => {
            if (this.activeTouchId == args.touchId) {
                this.activeTouchId = null;
                this.scrollAnimation.onTouchEnd(this.screenPointtoLocalPoint(args.position));
            }
        });
        this.outOfLeaderboardInteraction = this.getOrCreateInteraction(this.leaderboardBackground.getOutOfLeaderboardTouchZone().getSceneObject());
        this.outOfLeaderboardInteraction.isFilteredByDepth = true;
        this.outOfLeaderboardInteraction.onTap.add(() => {
            if (this.isLeaderboardOpened) {
                this.hide();
            }
        });
    }
    screenPointtoLocalPoint(screenPosition) {
        return this.screenTransform.screenPointToLocalPoint(screenPosition);
    }
    isInTouchZone(screenPos) {
        return this.screenTransform.containsScreenPoint(screenPos);
    }
    getOrCreateInteraction(sceneObject) {
        let interaction = sceneObject.getComponent('Component.InteractionComponent');
        if (isNull(interaction)) {
            interaction = this.destructionHelper.createComponent(sceneObject, 'Component.InteractionComponent');
            interaction.setCamera((0, ComponentUtils_1.findParentComponent)(sceneObject, 'Component.Camera'));
            interaction.isFilteredByDepth = false;
            interaction.setMinimumTouchSize(0);
            const rmv = sceneObject.getComponent('Component.Image');
            if (!rmv) {
                // an invisible render mesh visual for proper touch handling by InteractionComponent
                const image = this.destructionHelper.createComponent(sceneObject, 'Component.Image');
                image.stretchMode = StretchMode.Stretch;
                interaction.addMeshVisual(image);
            }
        }
        return interaction;
    }
    initialiseFallback() {
        this.isFallbackInitialised = true;
        this.fallbackEntriesParent = this.destructionHelper.createSceneObject(this.entriesParent.getParent(), "fallbackEntries");
        (0, ComponentUtils_1.getOrCreateScreenTransform)(this.fallbackEntriesParent, this.destructionHelper);
        for (let i = 0; i < 3; i++) {
            const currentEntriesCount = this.fallbackEntriesParent.children.length;
            const topRelativeLimit = LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRIES_LIMIT_TOP_POSITION_Y;
            const bottomPosition = topRelativeLimit - (currentEntriesCount + 1) * LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y;
            const topPosition = topRelativeLimit - currentEntriesCount * LeaderboardConstants_1.LeaderboardConstants.FullLeaderboard.ENTRY_SIZE_Y;
            const entry = this.leaderboardEntryFallbackPrefab.instantiate(this.fallbackEntriesParent);
            entry.enabled = true;
            const leaderboardFallbackEntry = entry.getComponent("ScriptComponent");
            leaderboardFallbackEntry.setEntryIndex(i);
            const screenTransform = entry.getComponent("ScreenTransform");
            screenTransform.anchors.top = topPosition;
            screenTransform.anchors.bottom = bottomPosition;
        }
        (0, ComponentUtils_1.setRenderOrderRecursivelyRelativeToParent)(this.fallbackEntriesParent, this.renderOrder);
        (0, ComponentUtils_1.setRenderLayerRecursively)(this.fallbackEntriesParent, this.entriesParent.layer);
    }
    reset() {
        this.userRecords = [];
        this.currentUserRecord = null;
        this.entryWrappers = [];
        this.entriesParent.children.forEach((entryScObj) => {
            entryScObj.destroy();
        });
        this.setFallbackEnabled(true);
    }
}
exports.FullLeaderboard = FullLeaderboard;
//# sourceMappingURL=FullLeaderboard.js.map